package gov.va.med.mhv.phr.appointments.model;

import org.apache.commons.lang.StringUtils;
import org.displaytag.decorator.TableDecorator;
import org.displaytag.model.AbstractRowSorter;

/**
 * Defines a display table row sorter that sorts on multiple properties.
 * <p>
 * The sorter will take the given property name as the primary property for 
 * sorting.
 * When two rows are equal it will subsequently sort on a number of (other) 
 * specified properties in the order that they are specified, until either 
 * the two rows are no longer equal, or no more properties are given.
 * </p><p>
 * This class will sort the two rows on the decorated values. 
 * This means that the TableDecorator should return unformatted objects, for
 * the comparison to be meaningful, unless formatted objects preserve the
 * normal ordering. This is may be the case for all objects, 
 * like e.g. java.util.Date.
 * In that case the column should have a decorator associated with it.
 * </p> 
 *  
 * @author Rob Proper
 *
 */
@SuppressWarnings("deprecation")
/* 
 * Suppressing deprecation warnings, because the new displaytag 
 * library does not provide this capability. The displaytag library
 * uses has been customized and is under control of MHV.
 */
public abstract class MultiPropertyRowSorter extends AbstractRowSorter {
	
	public MultiPropertyRowSorter(String propertyName, 
		TableDecorator tableDecorator, boolean isAscending) 
	{
		super(propertyName, tableDecorator, isAscending);
	}
	
	/**
	 * Specifies the properties to also order on. 
	 * @return The array of properties to order on
	 */
	protected abstract SortOrder[] getSortOrder();

	/**
	 * Compare the values by the properties denoted by the sort order.
	 * The sort order is determined by calling {@link #getSortOrder()}.
	 * The primary property is always used first, 
	 * using {@link AbstractRowSorter#compareValues(java.lang.Object, 
	 * java.lang.Object)}.
	 * Subsequently, the properties denoted in the sort order are used to
	 * order the two given objects, while the objects are considered equal.     
	 */
	@Override
	protected int compareValues(Object object1, Object object2) {
		final String primaryProperty = getPropertyName();
		assert !StringUtils.isBlank(primaryProperty);
		// Order on the primary property first, in the normal manner
		int comparison = super.compareValues(object1, object2);
		// Order on the standard ordering properties next
		for (SortOrder sortOrder: getSortOrder()) {
			if (comparison != 0) {
				// While objects are equal try this next property
				break;
			}
			if (!primaryProperty.equals(sortOrder.getPropertyName())) {
				comparison = compareProperty(sortOrder, object1, object2);
			} // else: Do not need to redo the primary property: they're equal
		}
		return comparison;
	}
	
	/**
	 * Compares the values for the property denoted by the given sort order.
	 * The decorated property value are used when available; 
	 * when unavailable the original values will be used.
	 * The comparison result will be reversed if so denoted by the given sort 
	 * order. 
	 * @param order The sort information, i.e. the property and order. 
	 * @param object1 The first object to determine the first decorated value 
	 *        from to be used in the comparison.
	 * @param object2 The second object to determine the second decorated value 
	 *        from to be used in the comparison.
	 * @return The comparison value: 
	 *         When the order is ascending, -1, 
	 *         if the value of object1 is greater than the value of object2; 
	 *         0 if the values are equal; 
	 *         and 1 if the value of object1 is less than the value of object2; 
	 *         When the order is descending, 1, 
	 *         if the value of object1 is greater than the value of object2; 
	 *         0 if the values are equal; 
	 *         and -1 if the value of object1 is less than the value of object2. 
	 */
	protected int compareProperty(SortOrder order, Object object1, 
		Object object2) 
	{
		final String propertyName = order.getPropertyName();
		assert !StringUtils.isBlank(propertyName);
		// Try the comparing the decorated values
		// Note that if a decorated value cannot be found the original value
		// wil be used automatically.
		return reverseIfDescending(nullSafeCompare(getDecoratedValue(object1, 
			propertyName), getDecoratedValue(object2, propertyName)), 
			order.isAscending());
	}

	
	/**
	 * Reverses the comparison value, if isAscending is true.
	 * Use this when you want check a secondary column. 
	 * {@link AbstractRowSorter#reverseIfDescending(int)} always applies 
	 * to the primary property. 
	 * @param comparison The comparison value.
	 * @param isAscending Whether the comparison value denotes an ascending 
	 *        order.
	 * @return The value of comparison if isAscending is true; -comparison
	 *         otherwise.
	 */
	protected final int reverseIfDescending(int comparison, 
		boolean isAscending) 
	{
		//  
		return (isAscending) ? comparison : (-1 * comparison);
	}

	/**
	 * Defines a property to sort on.
	 * descending.
	 *
	 */
	protected static final class SortOrder {

		private final String propertyName;
		private final boolean isAscending;
		
		public SortOrder(String propertyName, boolean isAscending) {
			assert !StringUtils.isBlank(propertyName);
			this.propertyName = propertyName;
			this.isAscending = isAscending;
		}
		/**
		 * Denotes whether the order in which the associated property must be 
		 * sorted: ascending or descending.
		 * @return Returns the order.
		 */
		public boolean isAscending() {
			return isAscending;
		}

		/**
		 * Denotes the property to sort.
		 * @return Returns the propertyName.
		 */
		public String getPropertyName() {
			return propertyName;
		}

	}

}
